<?php
defined('BASEPATH') OR exit('No direct script access allowed');

use \Firebase\JWT\JWT;

class Webvue_stat extends API
{
	
	public function __construct()
	{
		parent::__construct();
		$this->load->model('model_api_webvue_stat');
	}

	/**
	 * @api {get} /webvue_stat/all Get all webvue_stats.
	 * @apiVersion 0.1.0
	 * @apiName AllWebvuestat 
	 * @apiGroup webvue_stat
	 * @apiHeader {String} X-Api-Key Webvue stats unique access-key.
	 * @apiPermission Webvue stat Cant be Accessed permission name : api_webvue_stat_all
	 *
	 * @apiParam {String} [Filter=null] Optional filter of Webvue stats.
	 * @apiParam {String} [Field="All Field"] Optional field of Webvue stats : id, icon, title, color, number.
	 * @apiParam {String} [Start=0] Optional start index of Webvue stats.
	 * @apiParam {String} [Limit=10] Optional limit data of Webvue stats.
	 *
	 *
	 * @apiSuccess {Boolean} Status status response api.
	 * @apiSuccess {String} Message message response api.
	 * @apiSuccess {Array} Data data of webvue_stat.
	 *
	 * @apiSuccessExample Success-Response:
	 *     HTTP/1.1 200 OK
	 *
	 * @apiError NoDataWebvue stat Webvue stat data is nothing.
	 *
	 * @apiErrorExample Error-Response:
	 *     HTTP/1.1 403 Not Acceptable
	 *
	 */
	public function all_get()
	{
		$this->is_allowed('api_webvue_stat_all', false);

		$filter = $this->get('filter');
		$field = $this->get('field');
		$limit = $this->get('limit') ? $this->get('limit') : $this->limit_page;
		$start = $this->get('start');

		$select_field = ['id', 'icon', 'title', 'color', 'number'];
		$webvue_stats = $this->model_api_webvue_stat->get($filter, $field, $limit, $start, $select_field);
		$total = $this->model_api_webvue_stat->count_all($filter, $field);
		$webvue_stats = array_map(function($row){
						
			return $row;
		}, $webvue_stats);

		$data['webvue_stat'] = $webvue_stats;
				
		$this->response([
			'status' 	=> true,
			'message' 	=> 'Data Webvue stat',
			'data'	 	=> $data,
			'total' 	=> $total,
		], API::HTTP_OK);
	}

		/**
	 * @api {get} /webvue_stat/detail Detail Webvue stat.
	 * @apiVersion 0.1.0
	 * @apiName DetailWebvue stat
	 * @apiGroup webvue_stat
	 * @apiHeader {String} X-Api-Key Webvue stats unique access-key.
	 * @apiPermission Webvue stat Cant be Accessed permission name : api_webvue_stat_detail
	 *
	 * @apiParam {Integer} Id Mandatory id of Webvue stats.
	 *
	 * @apiSuccess {Boolean} Status status response api.
	 * @apiSuccess {String} Message message response api.
	 * @apiSuccess {Array} Data data of webvue_stat.
	 *
	 * @apiSuccessExample Success-Response:
	 *     HTTP/1.1 200 OK
	 *
	 * @apiError Webvue statNotFound Webvue stat data is not found.
	 *
	 * @apiErrorExample Error-Response:
	 *     HTTP/1.1 403 Not Acceptable
	 *
	 */
	public function detail_get()
	{
		$this->is_allowed('api_webvue_stat_detail', false);

		$this->requiredInput(['id']);

		$id = $this->get('id');

		$select_field = ['id', 'icon', 'title', 'color', 'number'];
		$webvue_stat = $this->model_api_webvue_stat->find($id, $select_field);

		if (!$webvue_stat) {
			$this->response([
					'status' 	=> false,
					'message' 	=> 'Blog not found'
				], API::HTTP_NOT_FOUND);
		}

					
		$data['webvue_stat'] = $webvue_stat;
		if ($data['webvue_stat']) {
			
			$this->response([
				'status' 	=> true,
				'message' 	=> 'Detail Webvue stat',
				'data'	 	=> $data
			], API::HTTP_OK);
		} else {
			$this->response([
				'status' 	=> false,
				'message' 	=> 'Webvue stat not found'
			], API::HTTP_NOT_ACCEPTABLE);
		}
	}

	
	/**
	 * @api {post} /webvue_stat/add Add Webvue stat.
	 * @apiVersion 0.1.0
	 * @apiName AddWebvue stat
	 * @apiGroup webvue_stat
	 * @apiHeader {String} X-Api-Key Webvue stats unique access-key.
	 * @apiPermission Webvue stat Cant be Accessed permission name : api_webvue_stat_add
	 *
 	 * @apiParam {String} Icon Mandatory icon of Webvue stats. Input Icon Max Length : 100. 
	 * @apiParam {String} Title Mandatory title of Webvue stats. Input Title Max Length : 100. 
	 * @apiParam {String} Color Mandatory color of Webvue stats. Input Color Max Length : 10. 
	 * @apiParam {String} Number Mandatory number of Webvue stats. Input Number Max Length : 10. 
	 *
	 * @apiSuccess {Boolean} Status status response api.
	 * @apiSuccess {String} Message message response api.
	 *
	 * @apiSuccessExample Success-Response:
	 *     HTTP/1.1 200 OK
	 *
	 * @apiError ValidationError Error validation.
	 *
	 * @apiErrorExample Error-Response:
	 *     HTTP/1.1 403 Not Acceptable
	 *
	 */
	public function add_post()
	{
		$this->is_allowed('api_webvue_stat_add', false);

		$this->form_validation->set_rules('icon', 'Icon', 'trim|required|max_length[100]');
		$this->form_validation->set_rules('title', 'Title', 'trim|required|max_length[100]');
		$this->form_validation->set_rules('color', 'Color', 'trim|required|max_length[10]');
		$this->form_validation->set_rules('number', 'Number', 'trim|required|max_length[10]');
		
		if ($this->form_validation->run()) {

			$save_data = [
				'icon' => $this->input->post('icon'),
				'title' => $this->input->post('title'),
				'color' => $this->input->post('color'),
				'number' => $this->input->post('number'),
			];
			
			$save_webvue_stat = $this->model_api_webvue_stat->store($save_data);

			if ($save_webvue_stat) {
				$this->response([
					'status' 	=> true,
					'message' 	=> 'Your data has been successfully stored into the database'
				], API::HTTP_OK);

			} else {
				$this->response([
					'status' 	=> false,
					'message' 	=> cclang('data_not_change')
				], API::HTTP_NOT_ACCEPTABLE);
			}

		} else {
			$this->response([
				'status' 	=> false,
				'message' 	=> 'Validation Errors.',
				'errors' 	=> $this->form_validation->error_array()
			], API::HTTP_NOT_ACCEPTABLE);
		}
	}

	/**
	 * @api {post} /webvue_stat/update Update Webvue stat.
	 * @apiVersion 0.1.0
	 * @apiName UpdateWebvue stat
	 * @apiGroup webvue_stat
	 * @apiHeader {String} X-Api-Key Webvue stats unique access-key.
	 * @apiPermission Webvue stat Cant be Accessed permission name : api_webvue_stat_update
	 *
	 * @apiParam {String} Icon Mandatory icon of Webvue stats. Input Icon Max Length : 100. 
	 * @apiParam {String} Title Mandatory title of Webvue stats. Input Title Max Length : 100. 
	 * @apiParam {String} Color Mandatory color of Webvue stats. Input Color Max Length : 10. 
	 * @apiParam {String} Number Mandatory number of Webvue stats. Input Number Max Length : 10. 
	 * @apiParam {Integer} id Mandatory id of Webvue Stat.
	 *
	 * @apiSuccess {Boolean} Status status response api.
	 * @apiSuccess {String} Message message response api.
	 *
	 * @apiSuccessExample Success-Response:
	 *     HTTP/1.1 200 OK
	 *
	 * @apiError ValidationError Error validation.
	 *
	 * @apiErrorExample Error-Response:
	 *     HTTP/1.1 403 Not Acceptable
	 *
	 */
	public function update_post()
	{
		$this->is_allowed('api_webvue_stat_update', false);

		
		$this->form_validation->set_rules('icon', 'Icon', 'trim|required|max_length[100]');
		$this->form_validation->set_rules('title', 'Title', 'trim|required|max_length[100]');
		$this->form_validation->set_rules('color', 'Color', 'trim|required|max_length[10]');
		$this->form_validation->set_rules('number', 'Number', 'trim|required|max_length[10]');
		
		if ($this->form_validation->run()) {

			$save_data = [
				'icon' => $this->input->post('icon'),
				'title' => $this->input->post('title'),
				'color' => $this->input->post('color'),
				'number' => $this->input->post('number'),
			];
			
			$save_webvue_stat = $this->model_api_webvue_stat->change($this->post('id'), $save_data);

			if ($save_webvue_stat) {
				$this->response([
					'status' 	=> true,
					'message' 	=> 'Your data has been successfully updated into the database'
				], API::HTTP_OK);

			} else {
				$this->response([
					'status' 	=> false,
					'message' 	=> cclang('data_not_change')
				], API::HTTP_NOT_ACCEPTABLE);
			}

		} else {
			$this->response([
				'status' 	=> false,
				'message' 	=> 'Validation Errors.',
				'errors' 	=> $this->form_validation->error_array()
			], API::HTTP_NOT_ACCEPTABLE);
		}
	}
	
	/**
	 * @api {post} /webvue_stat/delete Delete Webvue stat. 
	 * @apiVersion 0.1.0
	 * @apiName DeleteWebvue stat
	 * @apiGroup webvue_stat
	 * @apiHeader {String} X-Api-Key Webvue stats unique access-key.
	 	 * @apiPermission Webvue stat Cant be Accessed permission name : api_webvue_stat_delete
	 *
	 * @apiParam {Integer} Id Mandatory id of Webvue stats .
	 *
	 * @apiSuccess {Boolean} Status status response api.
	 * @apiSuccess {String} Message message response api.
	 *
	 * @apiSuccessExample Success-Response:
	 *     HTTP/1.1 200 OK
	 *
	 * @apiError ValidationError Error validation.
	 *
	 * @apiErrorExample Error-Response:
	 *     HTTP/1.1 403 Not Acceptable
	 *
	 */
	public function delete_post()
	{
		$this->is_allowed('api_webvue_stat_delete', false);

		$webvue_stat = $this->model_api_webvue_stat->find($this->post('id'));

		if (!$webvue_stat) {
			$this->response([
				'status' 	=> false,
				'message' 	=> 'Webvue stat not found'
			], API::HTTP_NOT_ACCEPTABLE);
		} else {
			$delete = $this->model_api_webvue_stat->remove($this->post('id'));

			}
		
		if ($delete) {
			$this->response([
				'status' 	=> true,
				'message' 	=> 'Webvue stat deleted',
			], API::HTTP_OK);
		} else {
			$this->response([
				'status' 	=> false,
				'message' 	=> 'Webvue stat not delete'
			], API::HTTP_NOT_ACCEPTABLE);
		}
	}
	
}

/* End of file Webvue stat.php */
/* Location: ./application/controllers/api/Webvue stat.php */